' TwistyPassages
' Idea stolen from 'X16MAZES' game for the CommanderX16
' Adapted for CMM2 by William M Leue, 16-Jan-2024

option default integer
option base 1

' Constants
const CELLH = 25
const CELLV = 20
const MAXCOLS = mm.hres\CELLH
const MAXROWS = mm.vres\CELLV

const FWIDTH = 20
const FCOLOR = rgb(0, 128, 0)
const BGCOLOR = rgb(190, 184, 143)

const NO_PASSAGE  =0
const NEW_PASSAGE = 1
const OLD_PASSAGE = 2

' Keyboard Commands
const UP    = 128
const DOWN  = 129
const LEFT  = 130
const RIGHT = 131
const SPACE = 32
const ENTER = 13
const N     = 78
const R     = 82
const PGUP  = 136
const PGDN  = 137
const SELCT = asc("S")
const ESC   = 27

' Win text box
const SWX = 200
const SWY = 200
const SWW = 400
const SWH = 200

' Globals
dim cells(MAXROWS, MAXCOLS)
dim passcnt = 0
dim ucnt = 0
dim start(2)
dim px = 0
dim py = 0
dim pwidth = 0
dim pheight = 0
dim path$ = ""
dim ucolor = 0
dim running = 0
dim level = 0
dim win = 0
dim urow = 0
dim ucol = 0
dim last_level = 0
dim float t0 = 0.0
dim float et = 0.0
dim float tet = 0.0

' Main program
open "debug.txt" for output as #1
level = 1
OpenMaze level, ok
if not ok then end
InitMaze
DrawMaze
DrawMover urow, ucol
HandleEvents
end

' Open the specified level maze file
sub OpenMaze level, ok
  path$ = MakeFullPath$("level" + str$(level))
  ReadMaze path$, ok
end sub  

' Add the prefix and suffix to the maze filename
function MakeFullPath$(fn$)
  if instr(1, fn$, "MAZES/") = 0 then
    fn$ = "./MAZES/" + fn$
  end if
  if instr(1, fn$, ".maz") = 0 then cat fn$, ".maz"
  MakeFullPath$ = fn$
end function

' Read the maze file
sub ReadMaze path$, ok
  local row, col, buf$
  ok = 1 
  on error skip 1
  open path$ for input as #2
  if mm.errno <> 0 then
    cls
    print "Open ";path$;" error: ";mm.errmsg$
    ok = 0
    exit sub
  end if
  line input #2, buf$
  level = val(buf$)
  line input #2, buf$
  cat buf$, ","
  pwidth = val(field$(buf$, 1, ","))
  pheight = val(field$(buf$, 2, ","))
  SetLocation  
  line input #2, buf$
  cat buf$, ","
  start(1) = val(field$(buf$, 1, ","))
  start(2) = val(field$(buf$, 2, ","))
  urow = start(1) : ucol = start(2)
  passcnt = 0
  ucnt = 0
  for row = 1 to pheight
    line input #2, buf$
    cat buf$, ","
    for col = 1 to pwidth
      cells(row, col) = val(field$(buf$, col, ","))
      if cells(row, col) = 1 then inc passcnt
    next col
  next row
  close #2
end sub

' Initialize Params for Maze Running
sub InitMaze
  local row, col
  running = 1
  win = 0
  ucount = 0
  last_level = 0
  for row = 1 to pheight
    for col = 1 to pwidth
      if cells(row, col) = OLD_PASSAGE then
        cells(row, col) = NEW_PASSAGE
      end if
    next col
  next row
  urow = start(1) : ucol = start(2)
  PickColor
  t0 = 0.001*timer
  et = 0
end sub

' Set the location of the maze tlc
sub SetLocation
  px = mm.hres\2 - (pwidth*CELLH)\2
  py = mm.vres\2 - (pheight*CELLV)\2
end sub

' Pick a color for the mover
sub PickColor
  local float h, s, v
  local r, g, b
  s = 0.5
  v = 1.0
  h = 1.0*RandInt(0, 300.0)
  HSV2RGB(h, s, v, r, g, b)
  ucolor = rgb(r, g, b)
end sub

' Draw the Maze
sub DrawMaze
  local row, col, x, y, c, v
  color rgb(white), rgb(black)
  DrawFrame 1
  for row = 1 to pheight
    y = py + (row-1)*CELLV
    for col = 1 to pwidth
      x = px + (col-1)*CELLH
      v = cells(row, col)
      if (v = NEW_PASSAGE) or (v = OLD_PASSAGE) then
        c = rgb(black)
      else if v = NO_PASSAGE then
        c = BGCOLOR
      end if
      box x, y, CELLH, CELLV,, c, c
    next col
  next row
end sub

' Draw the Frame
sub DrawFrame inside
  local h, w, m$
  w = mm.hres : h = mm.vres
  box 0, 0, w, h, FWIDTH, FCOLOR
  if inside then
    box FWIDTH, FWIDTH, w-2*FWIDTH, h-2*FWIDTH,, BGCOLOR, BGCOLOR
  end if
  m$ = "LEVEL " + str$(level)
  text 30, 3, m$, "LT", 4,, rgb(white), -1
  text 31, 3, m$, "LT", 4,, rgb(white), -1
  m$ = "Traverse All Black Passages to Finish Level"
  text w-30, 3, m$, "RT", 4,, rgb(white), -1
  m$ = "'R' - Restart Lvl  'PgUp', 'PgDn', 'S' - choose Lvl  'Esc' - Quit"
  text 30, h-3, m$, "LB", 4,, rgb(white), -1
  m$ = "TWISTY LITTLE PASSAGES"
  text 4, 60, m$, "LTV", 4,, rgb(white), -1
end sub

' Draw the mover circle
sub DrawMover row, col
  local x, y
  if not running then exit sub
  x = px + (col-1)*CELLH + CELLH\2
  y = py + (row-1)*CELLV + CELLV\2
  circle x, y, CELLV\4,,, rgb(black), ucolor
  circle x, y, CELLV\4-1,,, rgb(black), ucolor
end sub

' Handle User Input
sub HandleEvents
  local z$ = INKEY$
  local cmd, row, col, prev_row, prev_col, ok
  local a$, save_level
  row = urow : col = ucol
  do
    do
      z$ = INKEY$
    loop until z$ <> ""
    cmd = asc(UCASE$(z$))
    select case cmd
      if not running then continue do
      case UP, DOWN, LEFT, RIGHT
        DoMove row, col, cmd
        DrawMover row, col
      case PGUP
        tlevel = level+1
        if TestLevel(tlevel) = 0 then
          m$ = "Level " + str$(tlevel) + " does not exist"
          text mm.hres-30, 3, m$, "RT", 4,, rgb(red), -1
          text mm.hres-29, 3, m$, "RT", 4,, rgb(red), -1
          pause 1000
          DrawFrame 0
        else
          level = tlevel
          OpenMaze level, ok
          InitMaze
          row = start(1) : col = start(2)
          DrawMaze
          DrawMover urow, ucol
        end if
      case PGDN
        if tlevel = 1 then continue do
        last_level = 0
        tlevel = level-1
        if TestLevel(tlevel) = 0 then
          m$ = "Level " + str$(tlevel) + " does not exist"
          text mm.hres-30, 3, m$, "RT", 4,, rgb(red), -1
          text mm.hres-29, 3, m$, "RT", 4,, rgb(red), -1
          pause 1000
          DrawFrame 0
        else
          level = tlevel
          OpenMaze level, ok
          InitMaze
          row = start(1) : col = start(2)
          DrawMaze
          DrawMover row, col
        end if
      case SELCT
        cls
        print "Enter level number: ";
        input "", a$
        save_level = level
        level = val(a$)
        OpenMaze level, ok
        if ok then
          InitMaze
          row = start(1) : col = start(2)
          DrawMaze
          DrawMover urow, ucol
        else
          print "Sorry, that level does not exist"
          print "Returning to previous level"
          pause 3000
          level = save_level
          OpenMaze level, ok
          InitMaze
          row = start(1) : col = start(2)
          DrawMaze
          DrawMover urow, ucol
        end if
      case N
        if win then
          if last_level then continue do
          inc level
          OpenMaze level, ok
          InitMaze
          DrawMaze
          row = start(1) : col = start(2)
          DrawMover row, col
        end if
      case R
        if running then
          InitMaze
          row = urow : col = ucol
          DrawMaze
          DrawMover urow, ucol
        end if
      case ESC
        cls
        end
    end select
  loop
end sub

' Test to see if a specified maze level exists
function TestLevel(tlevel)
  local tpath$
  TestLevel = 1
  tpath$ = "./Mazes/level" + str$(tlevel) + ".maz"
  on error skip 1
  open tpath$ for input as #3
  if mm.errno <> 0 then
    TestLevel = 0
    exit function
  end if
  close #3
end function  

' Make the move
sub DoMove row, col, dir
  local erow, ecol
  local cnt = 0
  select case dir
    case UP
      for erow = row-1 to 1 step -1
        if IsPassage(erow, col) then
          inc cnt
        else
          exit for
        end if
      next erow
    case DOWN
      for erow = row+1 to pheight
        if IsPassage(erow, col) then
          inc cnt
        else
          exit for
        end if
      next erow
    case LEFT
      for ecol = col-1 to 1 step -1
        if IsPassage(row, ecol) then
          inc cnt
        else
          exit for
        end if
      next ecol
    case RIGHT
      for ecol = col+1 to pwidth
        if IsPassage(row, ecol) then
          inc cnt
        else
          exit for
        end if
      next ecol
  end select
  if cnt > 0 then AnimateMove row, col, dir, cnt
end sub

' Return 1 if the cell is a passage (NEW or OLD)
function IsPassage(row, col)
  IsPassage = 1
  if cells(row, col) = NO_PASSAGE then
    IsPassage = 0
  end if
end function

' Animate the Move
sub AnimateMove row, col, dir, cnt
  local prev_row, prev_col, i, drow, dcol, x, y, nx, ny
  prev_row = row : prev_col = col
  select case dir
    case UP
      drow = -1 : dcol = 0
    case DOWN
      drow = 1  : dcol = 0
    case LEFT
      drow = 0  : dcol = -1
    case RIGHT
      drow = 0  : dcol = 1
  end select
  for i = 1 to cnt
    x = px + (prev_col-1)*CELLH
    y = py + (prev_row-1)*CELLV
    box x, y, CELLH, CELLV,, ucolor, ucolor
    if cells(row, col) = NEW_PASSAGE then
      inc ucount
      cells(row, col) = OLD_PASSAGE
    end if
    inc x, dcol*CELLH
    inc y, drow*CELLV
    inc row, drow : inc col, dcol
    inc prev_row, drow
    inc prev_col, dcol
    circle x+CELLH\2, y+CELLV\2, CELLV\4,,, rgb(black), ucolor
    pause 50
  next i
    box x, y, CELLH, CELLV,, ucolor, ucolor
    if cells(row, col) = NEW_PASSAGE then
      inc ucount
      cells(row, col) = OLD_PASSAGE
    end if
  if ucount >= passcnt then
    running = 0
    win = 1
    ShowWin
  end if
end sub

' Show Win
sub ShowWin
  local x, y, m$, tlevel
  local float nt = 0.0
  box SWX, SWY, SWW, SWH,, rgb(black), rgb(150, 255, 255)
  nt = timer
  et = 0.001*nt - t0
  inc tet, et
  x = SWX + SWW\2
  y = SWY + SWH\2
  m$ = "Level " + str$(level) + " Solved!"
  text x, y, m$, "CM", 5,, rgb(blue), -1
  tlevel = level+1
  if TestLevel(tlevel) = 0 then
    m$ = "You have solved the highest level!"
    text mm.hres-30, 3, m$, "RT", 4,, rgb(red), -1
    text mm.hres-29, 3, m$, "RT", 4,, rgb(red), -1
    last_level = 1
  else
    m$ = "Time for this Level: " + str$(et) + " sec"
    text x, y+40, m$, "CT",,, rgb(black), -1
    m$ = "Total Time All Levels Completed: " + str$(tet) + " sec"
    text x, y+60, m$, "CT",,, rgb(black), -1
    text x, y+80, "Press 'N' for Next Level", "CT",,, rgb(black), -1
  end if
end sub

' Convert an HSV value to its RGB equivalent
' The S and V values must be in range 0..1; the H value must
' be in range 0..360. The RGB values will be in range 0..255.
sub HSV2RGB h as float, s as float, v as float, r, g, b
  local float i, hh, f, p, q, t, x, c, rp, gp, bp
  c = v*s
  hh = h/60.0
  i = int(hh)
  f = hh - i
  p = v*(1-s)
  q = v*(1-s*f)
  t = v*(1-s*(1-f))
  x = c*(1.0 - hh MOD 2 - 1)
  
  select case i
    case 0
      rp = v : gp = t : bp = p
    case 1
      rp = q : gp = v : bp = p
    case 2
      rp = p : gp = v : bp = t
    case 3
      rp = p : gp = q : bp = v
    case 4
      rp = t : gp = p : bp = v
    case 5
      rp = v : gp = p : bp = q
  end select
  r = rp*255.0 : g = gp*255.0 : b = bp*255.0
end sub

' return a uniformly distributed random integer in the specified closed range
function RandInt(a as integer, b as integer)
  local integer v, c
  c = b-a+1
  do
    v = a + (b-a+2)*rnd()
    if v >= a and v <= b then exit do
  loop
  RandInt = v
end function

